/*:
 * @target MZ
 * @plugindesc v1.0 フルスクリーンON/OFFをオプションに追加（HS_LangSimple連動・自動保存/起動時反映）
 * @author HS
 * @base HS_LangSimple
 *
 * @help
 * ・オプションに「フルスクリーン」(ON/OFF)を追加します
 * ・設定はConfigに保存。起動時に自動反映します
 * ・HS_LangSimpleの言語インデックス(hs_langIndex)に連動してラベルを切替
 * ・タイトルから開いたオプションに表示する/しないを選べます
 * ・イベントから操作できるプラグインコマンド付き
 *
 * 【注意】
 * - ブラウザ環境ではユーザー操作の直後でないと全画面化が拒否される場合があります
 * - NW.js(通常のMZ実行/ビルド)では問題なく動作します
 *
 * @param showInTitle
 * @text タイトルのオプションに表示
 * @type boolean
 * @default true
 *
 * @param defaultOn
 * @text 既定値（初回起動）
 * @type boolean
 * @default false
 *
 * @param labels
 * @text ラベル（言語別）
 * @type struct<Lbl>[]
 * @default [{"code":"ja","index":"0","text":"フルスクリーン"},{"code":"en","index":"1","text":"Fullscreen"},{"code":"zh-CN","index":"2","text":"全屏模式"},{"code":"ko","index":"3","text":"전체 화면"}]
 *
 * @command set
 * @text フルスクリーン設定
 * @desc フルスクリーン状態を切り替えます（Configにも保存）
 * @arg state
 * @text 状態
 * @type select
 * @option ON
 * @value on
 * @option OFF
 * @value off
 * @option トグル切替
 * @value toggle
 */
/*~struct~Lbl:
 * @param code
 * @text BCP47コード（任意）
 * @type string
 * @default ja
 *
 * @param index
 * @text 言語インデックス（HS_LangSimpleと揃える）
 * @type number
 * @min 0
 * @default 0
 *
 * @param text
 * @text ラベル
 * @type string
 * @default フルスクリーン
 */

(() => {
  "use strict";

  const PN = document.currentScript.src.match(/([^/]+)\.js$/)[1];
  const PRM = PluginManager.parameters(PN);
  const SHOW_IN_TITLE = PRM.showInTitle === "true";
  const DEFAULT_ON    = PRM.defaultOn === "true";

  // ---- labels: [{index:number, text:string}]
  const parseJSON = (s,d)=>{ try{ return JSON.parse(s); }catch{ return d; } };
  const LABELS = (parseJSON(PRM.labels, []))
    .map(x => typeof x === "string" ? parseJSON(x, null) : x)
    .filter(o => o && o.text != null)
    .map(o => ({ index: Number(o.index ?? 0), text: String(o.text ?? "") }));

  // ---- Language index (from HS_LangSimple if present)
  const langIndex = () => Number(ConfigManager.hs_langIndex ?? 0);
  const labelText = () => {
    const li = langIndex();
    const hit = LABELS.find(l => l.index === li);
    if (hit) return hit.text;
    // fallback: English / Japanese / first
    const en = LABELS.find(l => l.index === 1);
    const ja = LABELS.find(l => l.index === 0);
    return (en?.text || ja?.text || "Fullscreen");
  };

  // ---- Fullscreen helpers (engine & browser両対応)
  function isFullscreen() {
    try {
      if (typeof Graphics?._isFullScreen === "function") return !!Graphics._isFullScreen();
    } catch(_){}
    const d = document;
    const nw = (window.nw && nw.Window && nw.Window.get && nw.Window.get());
    const nwFs = nw && (typeof nw.isFullscreen === "boolean") ? nw.isFullscreen : undefined;
    return !!(d.fullscreenElement || d.webkitFullscreenElement || d.mozFullScreenElement || nwFs);
  }

  function requestFullscreen() {
    if (Graphics && typeof Graphics._requestFullScreen === "function") return void Graphics._requestFullScreen();
    const e = document.documentElement;
    (e.requestFullscreen || e.webkitRequestFullscreen || e.mozRequestFullScreen || (()=>{})).call(e);
    if (window.nw && nw.Window && nw.Window.get) {
      const win = nw.Window.get();
      if (win && typeof win.enterFullscreen === "function") win.enterFullscreen();
      else if (win && typeof win.toggleFullscreen === "function" && !win.isFullscreen) win.toggleFullscreen();
    }
  }

  function cancelFullscreen() {
    if (Graphics && typeof Graphics._cancelFullScreen === "function") return void Graphics._cancelFullScreen();
    (document.exitFullscreen || document.webkitCancelFullScreen || document.mozCancelFullScreen || (()=>{})).call(document);
    if (window.nw && nw.Window && nw.Window.get) {
      const win = nw.Window.get();
      if (win && typeof win.leaveFullscreen === "function") win.leaveFullscreen();
      else if (win && typeof win.toggleFullscreen === "function" && win.isFullscreen) win.toggleFullscreen();
    }
  }

  function applyFullscreen(on) {
    if (on && !isFullscreen()) requestFullscreen();
    if (!on && isFullscreen()) cancelFullscreen();
  }

  // ---- ConfigManager hook（保存/既定値/ロード反映 & セッターで即適用）
  let _hsFull = DEFAULT_ON;

  Object.defineProperty(ConfigManager, "hs_fullscreen", {
    get(){ return !!_hsFull; },
    set(v){ _hsFull = !!v; applyFullscreen(_hsFull); },
    configurable: true
  });

  const _setDefaults = ConfigManager.setDefaults;
  ConfigManager.setDefaults = function(){
    _setDefaults.call(this);
    this.hs_fullscreen = !!DEFAULT_ON; // setter経由で即反映
  };

  const _applyData = ConfigManager.applyData;
  ConfigManager.applyData = function(cfg){
    _applyData.call(this, cfg);
    if (cfg && typeof cfg.hs_fullscreen === "boolean") this.hs_fullscreen = cfg.hs_fullscreen;
    else this.hs_fullscreen = !!DEFAULT_ON;
  };

  const _makeData = ConfigManager.makeData;
  ConfigManager.makeData = function(){
    const data = _makeData.call(this);
    data.hs_fullscreen = !!this.hs_fullscreen;
    return data;
  };

  // ---- 起動時に一応もう一度同期（保険）
  const _bootStart = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function(){
    _bootStart.call(this);
    applyFullscreen(!!ConfigManager.hs_fullscreen);
  };

  // ---- Optionsに項目追加（言語連動ラベル）
  function titleInvokedOptions(){
    // HS_LangSimpleがセットするフラグを利用（なければfalse）
    return !!window.HS_TitleInvokedOptions;
  }

  const _addGeneralOptions = Window_Options.prototype.addGeneralOptions;
  Window_Options.prototype.addGeneralOptions = function(){
    _addGeneralOptions.call(this);
    if (!SHOW_IN_TITLE && titleInvokedOptions()) return; // タイトルから開いたオプションでは非表示
    // ブラウザでフルスクリーン不可の場合は出さない（任意）
    const canFS = document && (document.fullscreenEnabled !== false);
    if (canFS || Utils.isNwjs()) {
      this.addCommand(labelText(), "hs_fullscreen");
    }
  };

  // 表示ラベルはシーン再生成時に自動で切り替わるが、念のためrefreshでも更新
  const _makeCommandList = Window_Options.prototype.makeCommandList;
  Window_Options.prototype.makeCommandList = function(){
    _makeCommandList.call(this);
    // 既に追加済みのコマンド名を、最新の言語ラベルに差し替え
    for (const it of this._list) {
      if (it && it.symbol === "hs_fullscreen") it.name = labelText();
    }
  };

  // ---- オプションの値取得/設定
  const _statusText = Window_Options.prototype.statusText;
  Window_Options.prototype.statusText = function(index){
    const symbol = this.commandSymbol(index);
    if (symbol === "hs_fullscreen") return this.booleanStatusText(ConfigManager.hs_fullscreen);
    return _statusText.call(this, index);
  };

  const _processOk = Window_Options.prototype.processOk;
  Window_Options.prototype.processOk = function(){
    const index = this.index();
    const symbol = this.commandSymbol(index);
    if (symbol === "hs_fullscreen") {
      const v = !ConfigManager.hs_fullscreen;
      this.changeValue("hs_fullscreen", v); // setterで即適用＆保存はEsc/決定時
      SoundManager.playCursor();
      return;
    }
    _processOk.call(this);
  };

  // ---- プラグインコマンド
  PluginManager.registerCommand(PN, "set", args => {
    const mode = String(args.state || "toggle");
    if (mode === "on")  ConfigManager.hs_fullscreen = true;
    else if (mode === "off") ConfigManager.hs_fullscreen = false;
    else ConfigManager.hs_fullscreen = !ConfigManager.hs_fullscreen;
    ConfigManager.save();
  });

})();
